(* ::Subsubsection:: *)
(*SchmidtCassegrainCBefore package*)

(* ::Text:: *)
(*This impements the Schmidt Cassegrain telescope with a corrector in front of the Cassegrain combination*)

(* ::Text:: *)
(*Chapter*)

(* ::Text:: *)
(*This file should be in the following path*)

(* ::Text:: *)
(*FileNameJoin[{$UserBaseDirectory, "Applications/GeometricOptics/SchmidtCassegrainCBefore"}]*)

BeginPackage["GeometricOptics`Packages`SchmidtCassegrainCBefore`", {"GeometricOptics`"}]

Options[SchmidtCassegrainCBefore] = {OutputType->"Report", OutputLevel->"Full"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the evaluating notebook
					"Basic", gives the list of results
					"Values" or any other directive, gives the list of values (no names of variables)
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
*)
SchmidtCassegrainCBefore::usage="SchmidtCassegrainCBefore calculates parameters for a Schmidt Cassegrain telescope with a corrector in front of the Cassegrain combination.";

SchmidtCassegrainCBeforeInputPanel::usage = "SchmidtCassegrainCBeforeInputPanel gives the GUI panel to work with SchmidtCassegrainCBefore";

$SchmidtCassegrainCBeforeInputVariables = {"f1", "ft", "em", "Ni", "thick", "\[Delta]", "diam", "\[Theta]"};

Begin["`Private`"]

SchmidtCassegrainCBefore[{f1_, ft_, em_, Ni_, thick_, delta_, theta_}, opts___]:= SchmidtCassegrainCBefore[f1, ft, em, Ni, thick, diam, delta, theta, opts];

SchmidtCassegrainCBefore[f1_, ft_, em_, Ni_, thick_, diam_, delta_, theta_, OptionsPattern[]] := 
 Module[{fact, \[Alpha], M, R2, D10, D11, D12, D20, D21, D22, D30, D31, D32, a41, a21, Rc, sol1, fun1, sag1, a42, K22, a22, Rc2,
		 fun2, sag2, a43, K13, K23, a23, Rc3, sol3, fun3, sag3, outputs, inPanel, outPanel},
 
		If[ArgumentsQ["SchmidtCassegrainCBefore", {$SchmidtCassegrainCBeforeInputVariables, {f1, ft, em, Ni, thick, diam, delta, theta}}],
			AppendTo[$ExamplesStack, 
					 <|"PackageID" -> 10, "PackageName" -> "SchmidtCassegrainCBefore", 
					   "Arguments" ->  <|"f1" -> f1, "ft" -> ft, "em" -> em, "Ni" -> Ni, "thick" -> thick, "diam" -> diam, "\[Delta]" -> delta, "\[Theta]" -> theta|>|>];
			$ExamplesStack = DeleteExampleDuplicates[$ExamplesStack];
	
			fact = 0.866;
			\[Alpha] = em/f1;
			M = ft/f1;
			\[Beta] = (M - \[Alpha])/(M + 1);
			\[Gamma] = (2 M (1 + \[Alpha]))/(1 - M^2);
			R2 = \[Gamma] f1;
			D10 = -((1 + M^3 + \[Alpha] - M^2 (1 + \[Alpha]))/(32 f1^3 M^3));
			D11 = -(1/(32 f1^3));
			D12 = ((-1 + M)^3 (1 + \[Alpha]))/(32 f1^3 M^3 (1 + M));
			D20 = (M + \[Alpha] - M^3 (-1 + delta) - delta - \[Alpha] delta + M^2 (\[Alpha] (-1 + delta) + delta))/(8 f1^2 M^3);
			D21 = -(delta/(8 f1^2));
			D22 = ((-1 + M)^3 (M + \[Alpha] (-1 + delta) + delta))/(8 f1^2 M^3 (1 + M));
			D30 = 1/(8 f1 M^3 (1 + \[Alpha])) (M^4 + 2 M (\[Alpha] (-1 + delta) + delta) - 
					(\[Alpha] (-1 + delta) + delta)^2 + 
					M^2 (1 + \[Alpha]) (-1 + delta) (1 + \[Alpha] (-1 + delta) + delta) - 
					M^3 (4 - 2 delta + delta^2 + \[Alpha] (2 - 2 delta + delta^2)));
			D31 = -(delta^2/(8 f1));
			D32 = ((-1 + M)^3 (M + \[Alpha] (-1 + delta) + delta)^2)/(8 f1 M^3 (1 + M) (1 + \[Alpha]));
			
			a41 = -(D10/(-1 + Ni));
			a21 = -(1/2) a41 fact^2 diam^2;
			Rc = 1/(2 a21);
			
			TotalAberrations[{10^9, -(1/(a4 (fact^2)(diam^2))), -2 f1, \[Gamma] f1}, {thick, delta f1, -\[Beta] f1}, 
								{{1, Ni, 1, -1, 1}}, {0, {a4, as}, 0, 0}, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}];
			
			sol1 = FindRoot[GOaberration[[1]] == 0, {a4, a41}];
			fun1 = -(1/2) ReplaceAll[a4, sol1] fact^2 diam^2 Global`y^2 + ReplaceAll[a4, sol1] Global`y^4;
			sag1 = ReplaceAll[fun1, {Global`y -> fact diam}];

			outputs = { {"Primary radius", -2 f1},
						{"Secondary radius", R2},
						{"Secondary diameter", (1 - \[Beta]) diam},
						{"Distance between primary and secondary", -\[Beta] f1},
						{"Obstruction coefficient", 1 - \[Beta]},
						{"Elimination of spherical aberration with spherical mirrors", ""},
						{"Equation of corrector", TraditionalForm[fun1]},
						{"Curvature radius of the corrector at the vertex", -(1/(ReplaceAll[a4, sol1] fact^2 diam^2))},
						{"Sagitta of corrector", sag1},
						{"Total spherical aberration", GOaberration[[1]] /. sol1},
						{"Total coma", GOcoma[[1]] /. sol1},
						{"Total astigmatism", GOastigmatism[[1]] /. sol1},
						{"Image heigth", GOimageHeight[[1, GOn]] /. sol1},
						{"Focal length", GOfocalLength[[1]] /. sol1}};
			
			a42 = (D12 D20 - D10 D22)/(-D22 + D22 Ni);
			K22 = -(D20/D22);
			a22 = -(1/2) a42 fact^2 diam^2;
			Rc2 = 1/(2 a22);
			TotalAberrations[{10^9, -(1/(a42 (fact^2)(diam^2))), -2 f1, \[Gamma] f1}, {thick, delta f1, -\[Beta] f1}, 
								{{1, Ni, 1, -1, 1}}, {0, {a42, as}, 0, K22}, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}];

			fun2 = 1/(2 Rc2) Global`y^2 + a42 Global`y^4;
			sag2 = fun2 /. {Global`y -> fact diam};
			
			outputs = Join[outputs,
							{{"Elimination of spherical aberration and coma", ""},
							 {"Corrector equation", TraditionalForm[fun2]},
							 {"Conic constant of secondary mirror", K22},
							 {"Sagitta of corrector", sag2},
							 {"Curvature radius at the vertex", Rc},
							 {"Total spherical aberration", GOaberration[[1]]},
							 {"Total coma", GOcoma[[1]]},
							 {"Astigmatism", GOastigmatism[[1]]},
							 {"Image heigth", GOimageHeight[[1, GOn]] /. sol1},
							 {"Focal length", GOfocalLength[[1]]}}];
			
			a43 = (-D20 D32 - 4 D12 D30 f1 delta + 4 D10 D32 f1 delta + 4 D12 D20 f1^2 delta^2 + D22 (D30 - 4 D10 f1^2 delta^2))/(4 f1 (-1 + Ni) delta (-D32 + D22 f1 delta));
			K13 = (8 (D22 D30 - D20 D32) f1^2)/(delta (-D32 + D22 f1 delta));
			K23 = (D30 - D20 f1 delta)/(-D32 + D22 f1 delta);
			a23 = -(1/2) a43 fact^2 diam^2;
			Rc3 = 1/(2 a23);
			
			TotalAberrations[{10^9, -(1/(a4 (fact^2) (diam^2) )), -2 f1, \[Gamma] f1}, {thick, delta f1, -\[Beta] f1}, 
							  {{1, Ni, 1, -1, 1}}, {0, {a4, as}, K1, K2}, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}];

			sol3 = FindRoot[{GOaberration[[1]] == 0, GOcoma[[1]] == 0, GOastigmatism[[1]] == 0}, {a4, a43}, {K1, K13}, {K2, K23}];

			fun3 = -(1/2) ReplaceAll[a4, sol3] fact^2 diam^2 Global`y^2 + ReplaceAll[a4, sol3] Global`y^4;
			sag3 = fun3 /. {Global`y -> 1/2 fact diam};
			
			outputs = Join[outputs,
							{{"Elimination of spherical aberration, coma and astigmatism", ""},
							 {"Conic constant of primary", ReplaceAll[K1, sol3]},
							 {"Conic constant of secondary", ReplaceAll[K2, sol3]},
							 {"Corrector equation", TraditionalForm[fun3]},
							 {"Sagitta of corrector", sag3},
							 {"Curvature radius at the vertex", -(1/(ReplaceAll[a4, sol3] (fact^2) (diam^2) ))},
							 {"Total spherical aberration", GOaberration[[1]] /. sol3},
							 {"Total coma", GOcoma[[1]] /. sol3},
							 {"Total astigmatism", GOastigmatism[[1]] /. sol3},
							 {"Image heigth", GOimageHeight[[1, GOn]] /. sol1},
							 {"Focal length", GOfocalLength[[1]] /. sol3}}];

			(* defines the two panels, input parameters and output values *)
			inPanel = Grid[{{"Focal length of the primary mirror" , "f1", f1},
							{"Total focal length", "ft", ft},
							{"Back distance", "em", em},
							{"Refractive index of the corrector", "Ni", Ni},
							{"Thickness of the corrector", "thick", thick},
							{"Fraction of distance corrector-primary in focal unity", "delta", delta},
							{"Diameter of primary mirror", "diam", diam},
							{"Field angle  in degree", "theta", theta}},
							Alignment -> {{Left, Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"InputParameterBottom"}];
  
			outPanel = Grid[outputs, 
							Alignment -> {{Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"OutputValueBottom"}];
				
			(* generates the type of output required *)
			Switch[OptionValue[OutputType],
					"Report",
					GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "SchmidtCassegrainCBefore" &], "Description"], 
												"date" -> DateString[], 
												"function" -> "SchmidtCassegrainCBefore", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

					"Print",
					CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "SchmidtCassegrainCBefore" &], "Description"], 
													"date" -> DateString[], 
													"function" -> "SchmidtCassegrainCBefore", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
					"Basic",
					CellPrint[TextCell[
					TemplateApply[$BasicTemplate, 
									Join[<| "outputlevel" -> OptionValue[OutputLevel],
											"inputs" -> {{"f1" , f1},
														 {"ft", ft},
														 {"em", em},
														 {"Ni", Ni},
														 {"thick", thick},
														 {"delta", delta},
														 {"diam", diam},
														 {"\[Theta]", theta}},
											"outputs" -> outputs |>]], "Output"]],
					_,
					CellPrint[TextCell[DeleteCases[outputs[[All, 2]], Alternatives["", Style[___]]], "Output"]]],

							
			(* Arguments are not correct *)
			MessageDialog["SchmidtCassegrainCBefore not executed, the number or the type of the arguments may be incorrect.", WindowTitle->"Warning: example not generated"];]];

SchmidtCassegrainCBeforeInputPanel[]:=
DynamicModule[{outputtype = "Report", package = "SchmidtCassegrainCBefore"},
	examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
						Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
    examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
	example = "Browse...";
	Panel[Column[{	DynamicWrapper[Style[NameFromPackageName[package] <> " Input Panel", "Subsection"], 
									If[NumericQ[example], {f1, ft, em, Ni, thick, delta, diam, theta} = ReplaceAll[$SchmidtCassegrainCBeforeInputVariables, examplesAll[[example]]]]],
					Style["Insert values for each argument, then use Evaluate to run "<>package<>" function", "Text"],
					Grid[{{"Focal length of the primary mirror" , "f1", Tooltip[InputField[Dynamic[f1], Alignment -> Center], "Insert the value of focal length of the primary mirror"]},
						  {"Total focal length", "ft", Tooltip[InputField[Dynamic[ft], Alignment -> Center], "Insert the total focal length"]},
						  {"Back distance", "em", Tooltip[InputField[Dynamic[em], Alignment -> Center], "Insert the back distance"]},
						  {"Refractive index of the corrector", "Ni", Tooltip[InputField[Dynamic[Ni], Alignment -> Center], "Insert the refractive index of the corrector"]},
						  {"Thickness of the corrector", "thick", Tooltip[InputField[Dynamic[thick], Alignment -> Center], "Insert the thickness of the corrector"]},
						  {"Fraction of distance corrector-primary in focal unity", "\[Delta]", Tooltip[InputField[Dynamic[delta], Alignment -> Center], "Insert the \[Delta] coefficient"]},
						  {"Diameter of the primary mirror", "diam", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of diam"]},
						  {"Field angle in degrees", "\[Theta]", Tooltip[InputField[Dynamic[theta], Alignment -> Center], "Insert the value of \[Theta]"]}},
						Spacings -> {1, 0},
						Alignment -> {{Left, Left, Right}, Center}, 
						Dividers -> Center, 
						FrameStyle -> LightGray],
					OpenerView[{"Load an example from the archives (current session and saved DB)",
								Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center],
									 Spacer[5],
									 Button["Update list", (examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
																				Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
															examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], 
																						Alignment -> Center, 
																						Spacings -> {1, 1}, 
																						Dividers -> Center], 
																						FrameStyle -> LightGray]&, examplesAll];
															example = "Browse..."), Method -> "Queued"]}]}, 
								Alignment -> Center, 
								Spacings -> 1.5],
					Row[{"Define the type of output to generate", 
						 Spacer[5],
						 RadioButtonBar[Dynamic[outputtype], {"Report" -> Tooltip["Report", "Generates a new notebook reporting a summary of the calculation"], 
															  "Print" -> Tooltip["Print", "Print the table of the calculation done inside the current notebook"], 
															  "Basic" -> Tooltip["Basic", "Generate a list of computed output with label"], 
															  "Values" -> Tooltip["Values", "Return only the list of output values"]}]}],
					Row[{Button["Evaluate", ToExpression[package][Apply[Sequence, {f1, ft, em, Ni, thick, delta, diam, theta}], OutputType -> outputtype], Method -> "Queued"],
						 Button["Clear all", Map[Clear, Unevaluated[{f1, ft, em, Ni, thick, delta, diam, theta}]]]}]}, 
				Spacings -> 2, 
				Alignment -> Center],
	BaseStyle -> {InputFieldBoxOptions -> {FieldSize -> {15, 1}}}]];
			
  
  End[]
  EndPackage[]
  